/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 * Author Greg Hill <greg.hill@open-xchange.com>
 */

define('pgp_mail/util', [
    'io.ox/mail/util',
    'io.ox/core/capabilities',
    'settings!oxguard',
    'static/3rd.party/mailbuild.js'
], function (util, capabilities, settings, lib) {

    'use strict';

    function isEncryptEnabled() {
        if (window.oxguarddata && (window.oxguarddata.settings.oxguard || window.oxguarddata.action === 'secureReply')) return true;
        return false;
    }

    function isPGP(mail) {
        return isEncrypted(mail) || isSigned(mail);
    }

    function isEncrypted(mail) {
        if (mail.security_info && mail.security_info.encrypted) return true;
        if (mail.content_type && mail.content_type.indexOf('multipart/encrypted') > -1) return (true);
        try {
            if (mail.attachments && mail.attachments[0]) {
                if (mail.attachments[0].content.indexOf('---BEGIN PGP MESSAGE') > -1) {
                    mail.PGPInline = true;
                    return (true);
                }
            }
        } catch (e) {
            console.log('error checking body' + e);
        }
        return (false);
    }

    function isDecrypted(data) {
        if (_.isArray(data)) {
            var found = false;
            data.forEach(function (obj) {
                if (obj.security && obj.security.decrypted) found = true;
            });
            return found;
        }
        if (data.security && data.security.decrypted) return true;

        return false;
    }

    function isSigned(mail) {
        return mail.content_type === 'multipart/signed';
    }

    function getPGPInfo() {
        return $.when();
    }

    function buildFromModel(model) {
        var mail = new lib.mailbuild('multipart/encrypted; protocol="application/pgp-encrypted";');
        mail.addHeader({
            from: (model.get('from') || []).map(util.formatSender),
            to: (model.get('to') || []).map(util.formatSender),
            cc: (model.get('cc') || []).map(util.formatSender),
            bcc: (model.get('bcc') || []).map(util.formatSender),
            subject: model.get('subject')
        });
        mail.createChild('application/pgp-encrypted')
            .addHeader('content-description', 'PGP/MIME version identification')
            .setContent('Version: 1');
        mail.createChild('application/octet-stream; name="encrypted.asc"')
            .addHeader('content-description', 'PGP/MIME encrypted message')
            .addHeader('content-disposition', 'inline; filename="encrypted.asc"')
            .setContent(model.getContent());
        return mail;
    }

    return {
        isPGPMail: isPGP,
        isDecrypted: isDecrypted,
        isEncryptedMail: isEncrypted,
        isSignedMail: isSigned,
        getPGPInfo: getPGPInfo,
        isEncryptEnabled: isEncryptEnabled,
        builder: {
            fromModel: buildFromModel
        }
    };
});
