/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 * Author Greg Hill <greg.hill@open-xchange.com>
 */

define('pgp_local/promptKeyPaste', [
    'io.ox/core/extensions',
    'io.ox/backbone/views/modal',
    'io.ox/core/yell',
    'gettext!oxguard',
    'less!pgp_local/style'
], function (ext, ModalView, yell, gt) {

    'use strict';

    var POINT = 'oxguard/pgp_local/pasteKey';

    function openView(def) {
        new ModalView({
            async: true,
            point: POINT,
            title: gt('Add local key'),
            width: 450
        })
        .inject({
        })
        .build(function () {
        })
        .addCancelButton()
        .addButton({ label: gt('Add'), action: 'add' })
        .on('add', function () {
            var data = filterPrivate($('#keyData').val());
            if (data.length > 1) {
                def.resolve(data);
                this.close();
            } else {
                def.reject();
                yell('error', gt('No key data entered'));
                this.idle();
            }
        })
        .open();
    }

    var INDEX = 0;

    ext.point(POINT).extend(
        {
            index: INDEX += 100,
            id: 'paste',
            render: function () {
                this.$body.addClass('pastePrivate');
                var label = $('<label for="keyData">').append(gt('Paste a private key here:'));
                var keyData = $('<textarea type="text" id="keyData" class="og_guest_message"></textarea>');
                this.$body.append(
                    $('<div>').append(label.append(keyData))
                );
            }
        }, {
            index: INDEX += 100,
            id: 'fileUpload',
            render: function () {
                var fileDiv = $('<div class="fileSelect">');
                var label = $('<label for="fileSelectButton">').append(gt('or select a file location'));
                var fileSelect = $('<input type="file" id="privateFile" name="privateFile" style="display:none">');
                fileSelect.on('change', fileSelected);
                fileDiv.append(fileSelect);
                var fileButton = $('<button class="btn btn-default" id="fileSelectButton">').append(gt('Upload'));
                fileButton.on('click', function () {
                    fileSelect.click();
                });
                this.$body.append(
                    fileDiv.append(label).append(fileButton).append(fileSelect)
                );
            }
        }
    );

    function promptToPaste() {
        var def = $.Deferred();
        openView(def);
        return def;
    }

    // If begins with public key, remove that data
    function filterPrivate(data) {
        var i = data.indexOf('-----BEGIN PGP PRIVATE');
        if (i > 0) {
            data = data.substring(i);
        }
        return data;
    }

    function promptUpload() {
        yell('warning', gt('In order to receive encrypted emails, you must upload at least one public key to the server.  Please select a public key to upload from the key list, or upload another to the server'));
    }

    function fileSelected(e) {
        if (e.target.files && e.target.files.length > 0) {
            var file = e.target.files[0];
            var reader = new FileReader();
            reader.onload = function (f) {
                if (f && f.target && f.target.result) {
                    var data = f.target.result;
                    if (data.indexOf('BEGIN PGP PRIVATE') > 0) {
                        $('#keyData').val(data);
                    } else {
                        yell('error', gt('No private key found.'));
                    }
                }
            };
            reader.onerror = function (e) {
                console.log(e);
                yell('error', gt('Error reading file'));
            };
            reader.readAsText(file);
        }
    }

    return {
        promptToPaste: promptToPaste,
        promptUpload: promptUpload
    };
});
