/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 * Author Greg Hill <greg.hill@open-xchange.com>
 */

define('pgp_local/localHandler', [
    'gettext!oxguard',
    'oxguard/mail/checker',
    'pgp_local/localDecryptor',
    'pgp_local/sessionHandler'
], function (gt, checker, local, handler) {

    'use strict';

    var savedPassword;

    // Check if we have keys for session
    function checkKeys(baton, module) {
        var def = $.Deferred();
        local.checkLocal(baton.data.folder_id, baton.data.id, module)
        .done(function (sessionKeys) {
            local.checkIfHasKey(sessionKeys)
            .done(function () {
                baton.sessionKeys = sessionKeys;  // Store keys for later
                def.resolve();
            })
            .fail(function () {
                console.log('local key not found for session');
                def.reject();
            });
        });
        return def;
    }

    function decrypt(sessionKeys, password, def, baton, mail) {
        local.tryDecrypt(sessionKeys, password)
        .done(function (decryptedKey) {
            handler.saveDecryptedSession(decryptedKey)
            .done(function (authData) {
                if (mail) checker.pullAgain(decrypt, baton);
                baton.sessionKey = decryptedKey;
                def.resolve(authData.data.auth);
            });
        })
        .fail(function (e) {
            if (e === 'no key') {
                var err = {
                    code: 'local',
                    error: gt('No local key found')
                };
                checker.showError(err, baton);
                def.reject('no key');
            } else {
                //handleError (e, baton);
                def.reject();
            }
            console.log(e);
        });
    }

    // Do decryption of sessionKey to return
    function doLocal(baton, password, mail) {
        var def = $.Deferred();
        if (baton.sessionKeys) {  // If encrypted Session keys already stored, use them
            decrypt(baton.sessionKeys, password, def, baton, mail);
            return def;
        }
        local.checkLocal(baton.data.folder_id, baton.data.id)
        .done(function (sessionKeys) {
            decrypt(sessionKeys, password, def, baton, mail);
        })
        .fail(function (f) {
            if (f === 'no key') {
                var err = {
                    code: 'local',
                    error: gt('No local key found')
                };
                checker.showError(err, baton);
                def.reject('no key');
                return;
            }
            console.log(f);
            def.reject();

        });
        return def;
    }

    function savePassword(password, duration) {
        if (duration > 0) {
            savedPassword = password;
            var dur = duration * 60 * 1000;
            window.setTimeout(function () {
                savedPassword = undefined;
            }, dur);
        }
    }

    function fileFound(files, prompt, minSingleUse) {
        var def = $.Deferred();
        var baton;
        if (_.isArray(files)) { // files may be a list of files, or just a baton.  If array, create a baton
            baton = {
                data: files[0]
            };
        } else {
            baton = files;
        }
        checkKeys(baton, 'drive')
        .done(function () {
            console.log('for local file decrypt');
            if (baton.model && baton.model.get('guardAuth')) {   // If auth saved, then just attach to session and pull
                handler.saveDecryptedSession(baton.model.get('guardAuth'));
                def.resolve(baton.model.get('guardAuth'));
                return;
            }
            if (savedPassword) {
                doLocal(baton, savedPassword, false)
                .done(function (data) {
                    def.resolve(data);
                })
                .fail(function (e) {
                    def.reject(e);
                });
                return;
            }
            require(['oxguard/oxguard_core'], function (core) {
                if (prompt === undefined) {
                    prompt = gt('Enter %s security password', window.oxguarddata.productName);
                }
                core.getPassword(prompt, true)  //Get password
                .done(function (passdata) {
                    if (passdata.duration > -1 || minSingleUse) {
                        savePassword(passdata.password, passdata.duration);
                    }
                    doLocal(baton, passdata.password, false)
                    .done(function (data) {
                        def.resolve(data);
                    })
                    .fail(function (e) {
                        def.reject(e);
                    });
                })
                .fail(function () {
                    def.reject();
                });
            });
        })
        .fail(function () {
            def.reject();
        });
        return def;
    }

    // Check mail to see if it was encrypted with one of the private keys stored locally
    function mailFound(baton, location) {
        checkKeys(baton, 'mail')
        .done(function () {
            // We have a local key, just do local decrypt
            console.log('for local key decrypt');
            if (baton.model.get('guardAuth')) {   // If auth saved, then just attach to session and pull
                handler.saveDecryptedSession(baton.model.get('guardAuth'))
                .done(function () {
                    checker.pullAgain(true, baton);
                });
                return;
            }

            if (savedPassword) {
                doLocal(baton, savedPassword, true);
                return;
            }

            // Otherwise, mail prompt now
            var goFunction = function (password, duration) {
                doLocal(baton, password, true)
                .done(function () {
                    savePassword(password, duration);
                });
            };
            require(['oxguard/mail/oxguard_mail_password'], function (oxmail) {
                $(location).replaceWith(oxmail.passwordPrompt(baton, false, this, goFunction, true));
                baton.view.$el.find('.attachments').hide();
                if (!_.device('ios')) {
                    baton.view.$el.find('#oxgrpass').focus();
                }
                baton.view.$el.show();
            });
        })
        .fail(function () {
            console.log('not local key decrypt');
            // No local keys for this mail, try Guard server
            checker.pgpFound(baton, location, true);  // Call found again, but ignore private lookup
        });

    }

    return {
        mailFound: mailFound,
        fileFound: fileFound
    };

});
