/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/settings/userKeys', [
    'gettext!oxguard',
    'io.ox/core/tk/dialogs',
    'oxguard/core/og_http',
    'oxguard/core/passwords',
    'oxguard/oxguard_core',
    'io.ox/core/settings/util',
    'settings!oxguard',
    'less!oxguard/pgp/style'
], function (gt, dialogs, http, passwords, core, util, settings) {
    'use strict';

    var userKeyData = null;

    // List of users private keys
    function userKeys() {
        var pubdiv = $('<div class="userKeyDiv"/>');
        var tableHeader = $('<b>' + (settings.get('pgplocal') ? gt('Keys on Server') : gt('Your key list')) + '</b>');
        var refresh = $('<a href="#" style="margin-left:10px;">')
        .attr('title', gt('Refresh'))
        .append($('<i class="fa fa-refresh fa-spin" id="refreshuserkeys" aria-hidden="true">'))
            .click(function (e) {
                $(this).addClass('fa-spin');
                updateUserKeyTable().done(function () {
                    refresh.removeClass('fa-spin');
                });
                e.preventDefault();
            });
        var addKey = $('<a href="#" style="margin-left:10px;">')
        .attr('title', gt('Add user key'))
        .append($('<i class="fa fa-plus" id="adduserkeys" aria-hidden="true">'))
        .click(function (e) {
            ox.load(['oxguard/settings/views/uploadPrivateView']).then(function (view) {
                view.open();
            });
            e.preventDefault();
        });
        tableHeader.append(refresh).append(addKey);
        var tablediv = $('<div class="keytable">');
        var table = $('<table id="userkeytable">');
        pubdiv.append(tableHeader.append(tablediv.append(table)));
        updateUserKeys();
        return (pubdiv);
    }

    // List of locally stored private keys.  Not avail on server
    function localPrivateKeys() {
        if (!settings.get('pgplocal')) return;
        var localdiv = $('<div class="localKeyDiv" style="padding-top:20px;"/>');
        var tableHeader = $('<b>' + gt('Local Key List') + '</b>');
        var refresh = $('<a href="#" style="margin-left:10px;">')
        .attr('title', gt('Refresh'))
        .append($('<i class="fa fa-refresh" id="refreshlocalkeys" aria-hidden="true">'))
        .click(function (e) {
            $(this).addClass('fa-spin');
            updateUserKeys(true);
            e.preventDefault();
        });
        var addKey = $('<a href="#" style="margin-left:10px;">')
        .attr('title', gt('Add local keys'))
        .append($('<i class="fa fa-plus" id="addlocalkeys" aria-hidden="true">'))
        .click(function (e) {
            require(['pgp_local/register'], function (local) {
                local.promptPrivate()
                .done(function () {
                    updateUserKeys(true);
                });
            });
            e.preventDefault();
        });
        tableHeader.append(refresh).append(addKey);
        var tablediv = $('<div class="keytable">');
        var table = $('<table id="localuserkeytable">');
        localdiv.append(tableHeader.append(tablediv.append(table)));
        updateUserKeys(true);
        return (localdiv);
    }

    function updateUserKeys(local) {
        var def = $.Deferred();
        if (local) {
            $('#refreshlocalkeys').addClass('fa-spin');
            require(['pgp_local/keyDetails'], function (localKeyDetails) {
                window.setTimeout(function () {  // Give brief moment of spin
                    $('#refreshlocalkeys').removeClass('fa-spin');
                    $('#localuserkeytable').replaceWith(localKeyDetails.getLocalKeysTable());
                }, 200);
            });
            def.resolve();
            return;
        }
        $('#refreshuserkeys').addClass('fa-spin');
        updateUserKeyTable().done(function () {
            $('#refreshuserkeys').removeClass('fa-spin');
            def.resolve();
        });
        return def;
    }

    function getEightHex(fingerprint) {
        if (fingerprint) {
            return fingerprint.substring(fingerprint.length - 8);
        }
    }

    function updateUserKeyTable() {
        var deferred = $.Deferred();
        userKeyData = {};
        var params = '&userid=' + ox.user_id + '&cid=' + ox.context_id;
        // Send the request to the oxguard server for the encrypted content key to decrypt
        //http.get(ox.apiRoot + '/oxguard/pgp?action=getuserkeys', params)
        http.get(ox.apiRoot + '/oxguard/keys?action=getKeys', params)
        .done(function (grpdata) {
            grpdata = grpdata.data.keyRings ? grpdata.data.keyRings : [];
            var newtable = $('<table id="userkeytable">');
            var headers = $('<th>' + gt('Key ID') + '</th><th>' + gt('Private key') + '</th><th>' + gt('Current') + '</th><th>' + gt('Details') + '</th><th>' + gt('Delete') + '</th><th>' + gt('Download') + '</th><th>' + gt('Edit') + '</th>');
            if (grpdata.length === 0) {  // Keys deleted, no keys, open new key dialog
                window.oxguarddata.passcode = 'No Key';
                ox.load(['oxguard/settings/views/uploadPrivateView']).then(function (view) {
                    view.open();
                });
            }
            if (grpdata.length !== 0) {
                newtable.append(headers);

                for (var grp = 0; grp < grpdata.length; grp++) {
                    var rings = grpdata[grp].publicRing;
                    for (var i = 0; i < rings.keys.length; i++) {
                        var data = rings.keys[i];
                        userKeyData[data.id] = data; // Save in cache
                        var tr = (data.revoked || data.expired) ? $('<tr style="color:red;">') : $('<tr>');
                        if (data.revoked) $(tr).css('text-decoration', 'line-through');
                        var td1 = $('<td style="text-align:left;">');
                        if (data.masterKey === false) {
                            td1.append($('<i class="fa fa-level-up fa-rotate-90" style="padding-right:7px; vertical-align:bottom;">'));
                        }
                        td1.append($('<span style="padding-left:5px;">' + getEightHex(data.fingerPrint) + '</span>'));
                        tr.append(td1);
                        var td2 = $('<td>');
                        td2.append(data.hasPrivateKey ? ($('<i class="fa fa-check">').attr('title', gt('Has private key'))) : '');
                        tr.append(td2);
                        var td3 = $('<td>');
                        if (grpdata[grp].current) {
                            td3.append($('<i class="fa fa-check">').attr('title', gt('Current')));
                        } else {
                            var makecurrent = $('<input type="checkbox" class="makecurrent" value="' + data.id + '">')
                                .attr('title', gt('Make current'));
                            td3.append(makecurrent);
                        }

                        tr.append(td3);
                        var td4 = $('<td>');
                        var search = $('<a class="userpgpdetail" href="#">')
                            .attr('id', data.id)
                            .attr('title', gt('Details'))
                        .append($('<i class="fa fa-search" aria-hidden="true">'));
                        td4.append(search);
                        var td5 = $('<td>');
                        var del = $('<a href="#" class="delPriv">')
                            .attr('master', data.masterKey)
                            .attr('value', data.id)
                            .attr('private', (data.hasPrivateKey ? 'true' : 'false'))
                            .attr('title', gt('Delete'))
                        .append($('<i class="fa fa-trash" aria-hidden=true">'));
                        if (data.masterKey) {
                            td5.append(del);
                        }
                        var td6 = $('<td>');
                        var down = $('<a class="keydownload" href="#" >')
                            .attr('value', data.id)
                            .attr('title', gt('Download'))
                        .append($('<i class="fa fa-download" aria-hidden="true">'));
                        if (data.masterKey) td6.append(down);
                        var td7 = $('<td>');
                        var edit = $('<a class="keyedit" href="#" >')
                            .attr('value', data.id)
                            .attr('title', gt('Edit IDs'))
                        .append($('<i class="fa fa-pencil" aria-hidden="true">'));
                        if (data.masterKey && data.hasPrivateKey) td7.append(edit);
                        tr.append(td4).append(td5).append(td6).append(td7);
                        newtable.append(tr);
                    }
                    if ((grp + 1) < grpdata.length) {
                        var spacer = $('<tr>').append($('<td style="padding:1px;">').append('<hr style="border-top:1px solid #A59292; margin:5px;">'));
                        newtable.append(spacer);
                    }
                }

                $('#userkeytable').replaceWith(newtable);
                $('.userpgpdetail').on('click', function (e) {
                    var id = $(this).attr('id');
                    showLocalDetail(id);
                    e.preventDefault();
                });
                $('.keydownload').on('click', function (e) {
                    var id = $(this).attr('value');
                    ox.load(['oxguard/settings/views/downloadPrivateView']).then(function (view) {
                        view.open(id);
                    });
                    e.preventDefault();
                });
                $('.makecurrent').on('click', function (e) {
                    var id = $(this).attr('value');
                    makeCurrent(id);
                    e.preventDefault();
                });
                $('.delPriv').on('click', function (e) {
                    var id = $(this).attr('value');
                    var master = $(this).attr('master');
                    var privateAvail = $(this).attr('private');
                    if (privateAvail === 'false') {  // If no private avail, we are just deleting a public key
                        ox.load(['oxguard/settings/views/deletePublicView']).then(function (view) {
                            view.open(id, privateAvail);
                        });
                    } else {
                        ox.load(['oxguard/settings/views/deletePrivateView']).then(function (view) {
                            view.open(id, master);
                        });
                    }

                    e.preventDefault();
                });
                $('.keyedit').on('click', function (e) {
                    var id = $(this).attr('value');
                    ox.load(['oxguard/settings/views/editUserIdView']).then(function (view) {
                        view.open(id);
                    });
                    e.preventDefault();
                });
            } else {
                $('#userkeytable').replaceWith('<table id="userkeytable"><tr><td>' + gt('No Keys Found') + '</td></tr></table>');
            }
            deferred.resolve('OK');
            hideWait();
        })
        .fail(function () {
            hideWait();
        });
        return (deferred);
    }

    function showLocalDetail(id) {
        var data = getCacheData(id);
        ox.load(['oxguard/settings/views/privateDetailView']).then(function (view) {
            view.open(data);
        });
    }

    function makeCurrent(id) {

        wait();
        var params = '&userid=' + ox.user_id + '&cid=' + ox.context_id +
        '&keyid=' + id;
        // Send the request to the oxguard server for the encrypted content key to decrypt
        core.metrics('settings', 'change-current-key');
        http.put(ox.apiRoot + '/oxguard/keys?action=setCurrentKey', params, '')
        .done(function () {
            updateUserKeyTable();
        })
        .fail(function (e) {
            require(['io.ox/core/notifications'], function (notify) {
                notify.yell('error', e.responseText);
                console.log(e);
                hideWait();
            });
        });

    }

    function wait() {
        $('#og_wait').show();
    }

    function hideWait() {
        $('#og_wait').hide();
    }

    function getCacheData(id) {
        var data = userKeyData[id];
        return (data);
    }

    return {
        userKeys: userKeys,
        localKeys: localPrivateKeys
    };

});
