/* All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */

define('oxguard/mail/mailReplyActions', [
    'settings!oxguard',
    'io.ox/core/extensions'
], function (settings, ext) {

    'use strict';

    ext.point('io.ox/mail/compose/boot').extend({
        id: 'detectNewMail',
        index: 'first',
        perform: function (baton) {
            // returns here if any other type than new (e.g. replies etc)
            if (baton.data && baton.data.type && baton.data.type.toLowerCase() !== 'new') return;
            // returns here, if restorepoint or copy of draft is opened
            if (baton.data && baton.data.id) return;
            baton.config = _.extend({}, baton.config, { newmail: true });
        }
    });

    ext.point('io.ox/mail/compose/boot').extend({
        id: 'checkSecuritySettings',
        index: 1001,
        perform: function (baton) {
            if (baton.data.original && baton.data.original.security && baton.data.original.security.decrypted) {
                baton.view.securityModel.set('authToken', baton.data.original.security.authentication, true);
                baton.view.securityModel.set('encrypt', true);
            }
            if (baton.data.type === 'reply' || baton.data.type === 'replyall' || baton.data.type === 'forward') {
                checkReply(baton);
            }
            if (baton.data.type === 'edit') {
                checkEdit(baton);
            }
            if (baton.model.get('security').encrypt) {
                require(['oxguard/util'], function (util) {
                    if (!util.hasGuardMailCapability() || util.isGuest()) {
                        baton.view.toggleEncryption.forceEncryption();
                        disableRecipients(baton.view);
                    }
                });
                baton.model.save();
            }
        }
    });

    // If loading a draft email, need to set security settings same as before
    function checkEdit(baton) {
        var obj = baton.data.original;
        if (obj.headers && obj.headers['X-Security']) {
            var reply = obj.headers['In-Reply-To'] !== undefined;  // If this is editing a reply
            var securitySettings = _.isArray(obj.headers['X-Security']) ? obj.headers['X-Security'][0] : obj.headers['X-Security'];
            var settings = securitySettings.split(';');
            settings.forEach(function (setting) {
                var s = setting.split('=');
                var value = (s[1] ? (s[1].indexOf('true') > -1) : false);
                switch (s[0].trim()) {
                    case 'sign':
                        baton.view.securityModel.set('PGPSignature', value);
                        break;
                    case 'encrypt':
                        baton.view.securityModel.set('encrypt', value);
                        if (value === true && reply) {  // Check if we should enforce the secure reply here
                            if (window.oxguarddata.settings && window.oxguarddata.settings.secureReply) {
                                baton.view.toggleEncryption.forceEncryption();
                            }
                        }
                        break;
                    case 'pgpInline':
                        if (baton.view.securityModel.get('encrypt')) baton.view.securityModel.set('PGPFormat', (value ? 'inline' : 'mime'));
                        break;
                    default:
                        console.error('Unknown security parameter ' + s[0]);
                }
            });
            if (obj.security.authentication) {
                baton.view.securityModel.set('authentication', obj.security.authentication);
            }
            baton.model.unset('security_info');
        }
    }

    // When doing reply/forward, check essential security data moved to new model
    function checkReply(baton) {
        var obj = baton.data.original;
        var securityModel = baton.view.securityModel;
        if (obj.security && obj.security.decrypted) {
            securityModel.set('encrypt', true);
            var security = baton.model.get('security');
            security.encrypt = true;
            if (obj.security.authentication) {
                security.authentication = obj.security.authentication;
            }
            var headers = baton.model.get('headers') || {};
            baton.model.set('headers', _.extend(headers, { 'X-Guard-msg-ref': obj.id }));
            security.msgRef = obj.id;
            if (window.oxguarddata.settings && window.oxguarddata.settings.secureReply) {
                baton.view.toggleEncryption.forceEncryption();
            }
            baton.model.set('security', _.extend(baton.model.get('security'), { authentication: obj.security.authentication }));
        }
        baton.model.unset('security_info');

        // Check for default signed in reply emails
        if (settings.get('defaultSign')) {
            securityModel.set('PGPSignature', true);
        }
    }

    function disableRecipients(view) {
        view.$el.find('.recipient-actions').remove();
        view.$el.find('.tokenfield').css('background-color', '#e0e0e0');
        view.$el.find('.token-input').click(function (e) {
            $(e.currentTarget).blur();
            e.preventDefault();
            require(['gettext!oxguard', 'io.ox/core/notifications'], function (gt, notify) {
                notify.yell('error', gt('Recipients cannot be added to this encrypted reply'));
            });
        });
        view.$el.find('.twitter-typeahead').hide();
    }

});
