/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */

define('oxguard/core/tempPassword', [
    'io.ox/core/tk/dialogs',
    'oxguard/core/og_http',
    'oxguard/oxguard_core',
    'oxguard/core/passwordView',
    'settings!oxguard',
    'settings!io.ox/mail',
    'gettext!oxguard'
], function (dialogs, http, core, PasswordView, settings, mailSettings, gt) {

    'use strict';

    var firstprompt = false;

    // Prompt for initial password change
    function createOxGuardPasswordPrompt(baton, go, errors, oldpass, index, priorData) {
        if (firstprompt) return;
        firstprompt = true;
        var dialog = new dialogs.ModalDialog({ width: 450, center: true, enter: 'okpass' });
        var isnew = window.oxguarddata.new;
        var hasPin = window.oxguarddata.pin;
        var explain = '';
        if (isnew) {
            //#. %s product name
            dialog.header($('<h4>').text(gt('First %s Security Use', window.oxguarddata.productName)));
            explain = $('<div>');
            explain.append($('<p>').append(gt('You have been sent an email using %s to secure the contents', window.oxguarddata.productName)));
            explain.append($('<p>').append(gt('Please select a new password for use with %s.  This password will be used to open encrypted emails and files that may be sent to you.', window.oxguarddata.productName)));
        } else {
            //#. %s product name
            dialog.header($('<h4>').text(gt('First %s Security Use', window.oxguarddata.productName)));
            explain = $('<div><p>' + gt('Please change the initial temporary %s security password that was sent to you in a separate email.', window.oxguarddata.productName) + '</p></div>');
        }

        var passdiv = $('<div>').addClass('row-fluid');
        var newogpassword = new PasswordView.view({ 'id': 'newogpassword', 'class': 'password_prompt', 'validate': true });
        var td1 = $('<td>').append(newogpassword.getProtected());
        var newogpassword2 = new PasswordView.view({ 'id': 'newogpassword2', 'class': 'password_prompt' });
        var td2 = $('<td>').append(newogpassword2.getProtected());
        var hint = $('<td>');
        var table = $('<table class="og_password_prompt"/>');
        if (!isnew) {
            var oldPassword = new PasswordView.view({ 'id': 'oldogpassword', 'class': 'password_prompt', 'validate': false });
            table
            .append($('<tr>')
                .append($('<td class="pw">').append($('<label for="oldogpassword">').append(gt('Initial Password'))))
                .append($('<td>').append(oldPassword.getProtected())));
        }
        table
        .append($('<tr><td class="pw"><label for="newogpassword">' + gt('New Password') + ':</label></td></tr>').append(td1).append(hint))
        //#. Confirm new password
        .append($('<tr><td class="pw"><label for="newogpassword2">' + gt('Confirm New') + ':</label></td></tr>').append(td2).append('<td></td>'));
        passdiv.append(table);
        dialog.getBody().append(explain).append(passdiv);
        require(['oxguard/core/passwords'], function (pass) {
            pass.passwordCheck(newogpassword, hint);
            pass.passwordCompare(newogpassword2, newogpassword, hint);
        });
        if (hasPin) {
            var pinPrompt = $('<p>').append(gt('The sender assigned an additional PIN to the account.  This PIN must be entered before you can use this account.  Please enter it now.  Contact the sender directly if you don\'t yet have a pin.'));
            var pinTable = $('<table class="og_password_prompt">');
            var pinLabel = $('<td class="pw">').append('<label for="pinPrompt">' + gt('PIN') + ':</label>');
            var pinEntry = $('<input id="pinInput" name="pinInput" class="form-control password_prompt">');
            pinEntry.focus(function () {
                dialog.getBody().animate({ scrollTop: $('#pinInput').offset().top });
            });
            dialog.getBody().append('<hr/>').append(pinPrompt).append(pinTable.append($('<tr>').append(pinLabel).append($('<td>').append(pinEntry))));
        }
        if (window.oxguarddata.settings.noRecovery !== true) {
            var rule = $('<hr/>');
            var recovery = $('<p>' + gt('Please enter a secondary email address in case you need to reset your password.' + '</p>'));

            var table2 = $('<table class="og_password_prompt"/>');
            var email1 = $('<input id="recoveryemail" name="recoveryemail" class="form-control password_prompt"/>');
            var email2 = $('<input id="verifyemail" name="verifyemail" class="form-control password_prompt"/>');
            var hint2 = $('<td>');
            var row = $('<tr>').append($('<td class="pw"><label for="recoveryemail">' + gt('Email:') + '</label></td>')).append($('<td>').append(email1)).append(hint2);
            var row2 = $('<tr>').append($('<td class="pw"><label for="verifyemail">' + gt('Verify:') + '</label></td>')).append($('<td>').append(email2));
            table2.append(row).append(row2);
            email1.focus(function () {
                dialog.getBody().animate({ scrollTop: $('#recoveryemail').offset().top });
            });
            dialog.getBody().append(rule).append(recovery).append(table2);
            require(['oxguard/core/emailverify'], function (verify) {
                verify.setValidate(email1, hint2);
                verify.autoCompare(email1, email2, hint2);
            });
        } else {
            var warning = $('<p class="recoveryWarning">').append($('<b>').append(gt('Warning: This password for encryption cannot be restored or recovered in any way.  If forgotten, all encrypted data will be lost.')));
            dialog.getBody().append($('<hr/>')).append(warning);
        }
        if (errors !== undefined) {
            dialog.getBody().append('<div><span style="color: red;">' + errors + '</span></div>');
        }
        dialog
        .addPrimaryButton('okpass', gt('OK'), 'okpass')
        .addButton('cancel', gt('Cancel'), 'cancel')
        .on('cancel', function () {
            firstprompt = false;
        })
        .on('okpass', function () {
            firstprompt = false;
            var oldpass = $('#oldogpassword').val() === undefined ? '' : $('#oldogpassword').val();
            var pass1 = $('#newogpassword').val();
            var pass2 = $('#newogpassword2').val();
            var pin = hasPin ? $('#pinInput').val() : '';
            var emailaddr = $('input[name="recoveryemail"]').val() === undefined ? '' : $('input[name="recoveryemail"]').val();
            var verify = $('input[name="verifyemail"]').val() === undefined ? '' : $('input[name="verifyemail"]').val();
            if (emailaddr !== verify) {
                $('input[name="verifyemail"]').css('background-color', 'salmon');
                createOxGuardPasswordPrompt(baton, go, gt('Emails not equal'), oldpass);
                return;
            }
            if (pass1 !== pass2) {
                $('input[name="newogpassword2"]').css('background-color', 'salmon');
                createOxGuardPasswordPrompt(baton, go, gt('Passwords not equal'), oldpass);
                return;
            }
            var defaultSendAddress = $.trim(mailSettings.get('defaultSendAddress', ''));

            var userdata = {
                newpass: pass1,
                oldpass: oldpass,
                email: emailaddr,
                userEmail: defaultSendAddress ? defaultSendAddress : ox.user,
                user_id: ox.user_id,
                sessionID: ox.session,
                cid: ox.context_id,
                pin: pin
            };
            if ((emailaddr.length > 1) && (!validateEmail(emailaddr))) {
                createOxGuardPasswordPrompt(baton, go, gt('Enter new secondary Email address'), oldpass);
                return;
            }
            if (validateEmail(emailaddr)) userdata.email = emailaddr;
            require(['io.ox/core/notifications'], function (notifications) {
                http.post(ox.apiRoot + '/oxguard/login?action=changepass', '', userdata)
                .done(function (data) {
                    if (typeof data === 'string') data = $.parseJSON(data);
                    if (data.auth === undefined) {
                        if (data.error) {
                            createOxGuardPasswordPrompt(baton, go, data.error, oldpass, index, userdata);
                        } else {
                            require(['oxguard/core/errorHandler'], function (err) {
                                err.showError(data);
                            });
                        }
                        return;
                    }
                    if (data.auth.length > 20) {
                        $('#grdsettingerror').text(gt('Success'));
                        $('input[name="newogpassword"]').val('');
                        $('input[name="newogpassword2"]').val('');
                        window.oxguarddata.passcode = null;
                        window.oxguarddata.recoveryAvail = window.oxguarddata.recovery = !window.oxguarddata.settings.noRecovery;
                        core.saveAuthToSession(data.auth, -1).then(function () {
                            if (go) go(baton, pass1, index);
                        });
                        notifications.yell('success', gt('Password changed successfully'));
                        window.oxguarddata.pin = false;
                        updateSecondary();
                        window.oxguarddata.new = false;
                        // Complete, now offer tour
                        require(['io.ox/core/capabilities', 'io.ox/core/tk/wizard', 'oxguard/tour/main'], function (capabilities, Tour) {
                            if (capabilities.has('guard-mail') && !capabilities.has('guest')) { // only run tour if has guard-mail
                                Tour.registry.get('default/oxguard/createKeys').get('run')();
                            }
                        });

                    } else {
                        if (data.auth === 'Bad new password') {
                            notifications.yell('error', gt('New password must be at least %s characters long', data.minlength));
                            return;
                        }
                        if (data.auth === 'Bad password') {
                            //#. Bad, incorrect password
                            notifications.yell('error', gt('Bad password'));
                            return;
                        }
                        if (data.auth === 'Key not found') {
                            notifications.yell('error', gt('Encryption key not found'));
                            return;
                        }
                        notifications.yell('error', gt('Failed to change password'));
                    }
                })
                .fail(function () {
                    notifications.yell('error', gt('Failed to change password'));
                });
            });
        })
        .show(function () {
            if (priorData) {  // If error dialog, and reloaded, bring back old data
                $('#newogpassword').val(priorData.newpass);
                $('#newogpassword2').val(priorData.newpass);
                $('input[name="recoveryemail"]').val(priorData.email);
                $('input[name="verifyemail"]').val(priorData.email);
            }
            if (oldpass !== undefined) {
                $('#oldogpassword').val(oldpass);
                if (!_.device('ios')) {
                    $('#newogpassword').focus();
                }
            } else
            if (!_.device('ios')) {
                $('#oldogpassword').focus();
            }
            if (errors) {
                if ($('#recoveryemail').length > 0) {
                    dialog.getBody().animate({ scrollTop: $('#recoveryemail').offset().top });
                } else {
                    dialog.getBody().animate({ scrollTop: $('.recoveryWarning').offset().top });
                }

            }
            window.setTimeout(function () {
                var j = 0;
                $('.pw').each(function (i, v) {
                    var w = $(v).width();
                    if (w > j) j = w;
                });
                $('.pw').each(function (i, v) {
                    $(v).width(j);
                });
                var email = $('input[name="recoveryemail"]'); // Watch email entry for valid
                email.on('keyup', function () {
                    if (validateEmail(email.val())) {
                        email.css('background-color', 'white');
                    } else {
                        email.css('background-color', 'rgb(253, 187, 179)');
                    }
                });
            }, 0);
        });
    }

    // Valide legit email address
    function validateEmail(email) {
        var re = /^(([^<>()[\]\\.,;:\s@"]+(\.[^<>()[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
        return re.test(email);
    }

    function updateSecondary() {
        if ($('#currentsecondary').length > 0) {
            require(['oxguard/settings/pane'], function (settingsPage) {
                settingsPage.updateEmail();
            });
        }
    }

    return {
        createOxGuardPasswordPrompt: createOxGuardPasswordPrompt
    };

});
