/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.file.storage;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.Document;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.composition.AbstractDelegatingIDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.guard.api.GuardApi;
import com.openexchange.guard.file.storage.exceptions.GuardFileStorageExceptionCodes;
import com.openexchange.guard.internal.authentication.AuthenticationTokenHandler;
import com.openexchange.guard.internal.authentication.GuardAuthenticationToken;
import com.openexchange.mail.mime.MimeType2ExtMap;
import com.openexchange.session.Session;

/**
 * {@link DecryptingGuardAwareIDBasedFileAccess}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.8.3
 */
public class DecryptingGuardAwareIDBasedFileAccess extends AbstractDelegatingIDBasedFileAccess {

    private final GuardApi guardApi;
    private final Session session;
    private final EncryptedFileRecognizer encryptedFileRecognizer;
    private final GuardAuthenticationToken authenticationToken;

    private class GuardDocument extends Document {

        private final Document document;

        /**
         * Initializes a new {@link GuardDocument}.
         *
         * @param document The original {@link Document} to create a GuardDocument for
         * @throws OXException
         */
        public GuardDocument(Document document) throws OXException {
            super(document);
            this.document = document;
            //We do not know the size of the decrypted file yet
            setSize(-1);
            //Updating file data as well
            setFile(updateFile(getFile()));
        }

        /*
         * (non-Javadoc)
         *
         * @see com.openexchange.file.storage.Document#getData()
         */
        @Override
        public InputStream getData() throws OXException {
            return createGuardDecryptringInputStream(authenticationToken, document.getData());
        }

        /**
         * Signals that the InputStream returned by {@link GuardDocument#getData()} is not repetitive.
         *
         * @return false
         * @see com.openexchange.file.storage.Document#isRepetitive()
         */
        @Override
        public boolean isRepetitive() {
            return false;
        }
    }

    /**
     * Initializes a new {@link DecryptingGuardAwareIDBasedFileAccess}.
     *
     * @param fileAccess The underlying {@link IDBasedFileAccess} object to use.
     * @param guardApi The {@link GuardApi} instance used for accessing OX Guard.
     * @param session The user's session.
     * @param encryptedFileRecognizer A strategy for detecting encrypted files.
     * @param authenticationToken The {@link GuardAuthenticationToken} used for authentication.
     */
    public DecryptingGuardAwareIDBasedFileAccess(IDBasedFileAccess fileAccess, GuardApi guardApi, Session session, EncryptedFileRecognizer encryptedFileRecognizer, GuardAuthenticationToken authenticationToken) {
        super(fileAccess);
        this.guardApi = guardApi;
        this.session = session;
        this.encryptedFileRecognizer = encryptedFileRecognizer;
        this.authenticationToken = authenticationToken;
    }

    /**
     * Initializes a new {@link DecryptingGuardAwareIDBasedFileAccess}.
     * <p>
     * This will try to obtain the {@link GuardAuthenticationToken} from the given session's parameter.
     * </p>
     *
     * @param fileAccess The underlying {@link IDBasedFileAccess} object to use.
     * @param guardApi The {@link GuardApi} instance used for accessing OX Guard.
     * @param session The user's session.
     * @param encryptedFileRecognizer A strategy for detecting encrypted files.
     */
    public DecryptingGuardAwareIDBasedFileAccess(IDBasedFileAccess fileAccess, GuardApi guardApi, Session session, EncryptedFileRecognizer encryptedFileRecognizer) {
        this(fileAccess, guardApi, session, encryptedFileRecognizer, null);
    }

    /**
     * Creates a new InputStream which decrypts data using OX Guard
     *
     * @param authenticationToken An {@link GuardAuthenticationToken} to use, or null in order to obtain the token from the current session.
     * @param inputStream The stream which contains the data to be decrypted.
     * @return An decrypting InputStream
     * @throws OXException
     */
    private InputStream createGuardDecryptringInputStream(GuardAuthenticationToken authenticationToken, InputStream inputStream) throws OXException {
        if (authenticationToken == null) {
            authenticationToken = new AuthenticationTokenHandler().requireForSession(session);
        }
        return new GuardDecryptringInputStream(guardApi, authenticationToken, inputStream, session);
    }

    /**
     * Checks if the file is an encrypted document
     *
     * @param file The file to check
     * @return True, if the file is encrypted, false otherwise
     */
    private boolean isEncryptedGuardFile(File file) {
        return encryptedFileRecognizer.isEncrypted(file);
    }

    /**
     * Decrypts the given document
     *
     * @param document The document to decrypt
     * @return The decrypted document if the document is encrypted or the original document if the document is not encrypted
     * @throws OXException
     */
    private Document createDecryptedDocument(Document document) throws OXException {
        if (isEncryptedGuardFile(document.getFile())) {
            return markDecrypted(new GuardDocument(document));
        }
        return document;
    }

    /**
     * @param document
     * @return
     */
    private Document markDecrypted(GuardDocument document) {
        File file = document.getFile();
        encryptedFileRecognizer.markDecrypted(document);
        document.setFile(file);
        document.setMimeType(MimeType2ExtMap.getContentType(document.getName()));
        return document;
    }

    /**
     * Internal method to update/set Guard related meta data.
     *
     * @param file The meta data to update
     * @return The meta data
     */
    private File updateFile(File file) {
        //We do not know the size of the decrypted file yet
        file.setFileSize(-1);
        file.setFileMD5Sum("");
        file.setFileMIMEType(MimeType2ExtMap.getContentType(file.getFileName()));
        encryptedFileRecognizer.markDecrypted(file);
        return file;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.file.storage.composition.AbstractDelegatingIDBasedFileAccess#copy(java.lang.String, java.lang.String, java.lang.String, com.openexchange.file.storage.File, java.io.InputStream, java.util.List)
     */
    @Override
    public String copy(String sourceId, String version, String destFolderId, File update, InputStream newData, List<Field> modifiedFields) throws OXException {

        if (newData == null) {
            //If the caller did not provide data we fetch them from the original file
            newData = fileAccess.getDocument(sourceId, version);
        }
        newData = createGuardDecryptringInputStream(authenticationToken, newData);

        if (update == null) {
            update = fileAccess.getFileMetadata(sourceId, version);
        } else {
            File sourceFile = fileAccess.getFileMetadata(sourceId, version);
            if (update.getFileName() == null || update.getFileName().isEmpty()) {
                update.setFileName(sourceFile.getFileName());
            }
            update.setFileMIMEType(sourceFile.getFileMIMEType());
            update.setMeta(sourceFile.getMeta());
        }

        modifiedFields.addAll(encryptedFileRecognizer.markDecrypted(update));
        return fileAccess.copy(sourceId, version, destFolderId, update, newData, modifiedFields);
    }

    @Override
    public InputStream getDocument(String id, String version) throws OXException {
        return createDecryptedDocument(fileAccess.getDocumentAndMetadata(id, version)).getData();
    }

    @Override
    public InputStream getDocument(String id, String version, long offset, long length) throws OXException {
        Document document = fileAccess.getDocumentAndMetadata(id, version);
        if (isEncryptedGuardFile(document.getFile())) {
            try {
                //we cannot support random file access so we provide a PartialInputStream which allows to skip
                return new PartialInputStream(createDecryptedDocument(document).getData(), offset, length);
            } catch (IOException e) {
                throw GuardFileStorageExceptionCodes.IO_ERROR.create(e, e.getMessage());
            }
        }
        return fileAccess.getDocument(id, version, offset, length);
    }

    @Override
    public Document getDocumentAndMetadata(String id, String version) throws OXException {
        return createDecryptedDocument(fileAccess.getDocumentAndMetadata(id, version));
    }

    @Override
    public Document getDocumentAndMetadata(String id, String version, String clientEtag) throws OXException {
        return createDecryptedDocument(fileAccess.getDocumentAndMetadata(id, version, clientEtag));
    }

    @Override
    public InputStream optThumbnailStream(String id, String version) throws OXException {
        Document document = fileAccess.getDocumentAndMetadata(id, version);
        if (document != null) {
            File file = document.getFile();
            if (file != null) {
                if (!isEncryptedGuardFile(file)) {
                    return fileAccess.optThumbnailStream(id, version);
                }
            }
        }

        /* Either document/file is null or the file is encrypted and we cannot provide a thumbnail */
        return null;
    };

    @Override
    public File getFileMetadata(String id, String version) throws OXException {
        return updateFile(fileAccess.getFileMetadata(id, version));
    }
}
