/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.ant.tasks;

import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import com.openexchange.obs.api.Architecture;
import com.openexchange.obs.api.BuildServiceClient;
import com.openexchange.obs.api.BuildServiceException;
import com.openexchange.obs.api.Repository;

/**
 * Reads build service project information from a give project and sets properties according to read values.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public class ReadProjectSettings extends Task {

    private String url;
    private String login;
    private String password;
    private String projectName;
    private String repositoriesPropertyName;
    private String obsConfigPropertyName;
    private String archPropertyName;

    public ReadProjectSettings() {
        super();
    }

    public void setProject(String projectNames) {
        if (projectNames.length() != 0) {
            //will work with the first given project
            this.projectName = projectNames.split(",", 0)[0];
        }
    }

    public void setRepositoriesPropertyName(String repositoriesPropertyName) {
        this.repositoriesPropertyName = repositoriesPropertyName;
    }

    public void setObsConfigPropertyName(String obsConfigPropertyName) {
        this.obsConfigPropertyName = obsConfigPropertyName;
    }

    public void setLogin(String login) {
        this.login = login;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public void setArchPropertyName(String archPropertyName) {
        this.archPropertyName = archPropertyName;
    }

    @Override
    public void execute() throws BuildException {
        // check ant task arguments
        if (null == url || 0 == url.length()) {
            throw new BuildException("URL of build service is not defined.");
        }
        if (null == login || 0 == login.length()) {
            throw new BuildException("User for authenticating in build service is not defined.");
        }
        if (null == password || 0 == password.length()) {
            throw new BuildException("Password for authenticating in build service is not defined.");
        }
        if (null == projectName || 0 == projectName.length()) {
            throw new BuildException("Project in build service is not defined.");
        }
        if (null == repositoriesPropertyName || 0 == repositoriesPropertyName.length()) {
            throw new BuildException("repositoriesPropertyName is not defined.");
        }
        if (null == obsConfigPropertyName || 0 == obsConfigPropertyName.length()) {
            throw new BuildException("obsConfigPropertyName is not defined.");
        }
        if (null == archPropertyName || 0 == archPropertyName.length()) {
            throw new BuildException("archPropertyName is not defined.");
        }
        final BuildServiceClient bsc;
        try {
            bsc = new BuildServiceClient(url, login, password);
            Repository[] repos = bsc.getRepositoryList(projectName).getRepositories();
            StringBuilder sb = new StringBuilder();
            Architecture arch = null;
            for (Repository repository : repos) {
                sb.append(repository.getName());
                sb.append(',');
                arch = repository.getArch();
            }
            if (sb.length() != 0) {
                sb.setLength(sb.length() - 1);
            }
            log(repositoriesPropertyName + " => " + sb.toString(), Project.MSG_INFO);
            getProject().setInheritedProperty(repositoriesPropertyName ,sb.toString());
            String obsConfValue = bsc.getProjectConfig(projectName);
            Matcher m = Pattern.compile("(.*)\nMacros:", Pattern.DOTALL).matcher(obsConfValue);
            if (m.find()) {
                log(obsConfigPropertyName + " =>\n" + m.group(1), Project.MSG_INFO);
                getProject().setInheritedProperty(obsConfigPropertyName , m.group(1));
            } else {
                throw new BuildException("No config found for project " +  projectName);
            }
            if (null == arch) {
                throw new BuildException("Unable to determine architecture of project " + projectName);
            }
            log(archPropertyName + " => " + arch.toString(), Project.MSG_INFO);
            getProject().setInheritedProperty(archPropertyName, arch.toString());
        } catch (IOException e) {
            throw new BuildException(e.getMessage(), e);
        } catch (BuildServiceException e) {
            throw new BuildException(e.getMessage(), e);
        }
    }
}
