/*
 *
 *  Copyright (C) 2012 Open-Xchange, Inc. All rights reserved.
 *
 *  This program is proprietary software and licensed to you under Open-Xchange
 *  Inc's Software License Agreement.
 *
 *  In some countries OX, OX Open-Xchange, open-xchange and OXtender as well as
 *  the corresponding Logos OX Open-Xchange and OX are registered trademarks of
 *  the Open-Xchange, Inc. group of companies.
 *
 */

package com.openexchange.documentconverter;

import java.io.File;
import java.util.Date;
import java.util.HashMap;

/**
 * {@link ICacheEntry}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public interface ICacheEntry {

    /**
     * @return True if this is a valid cache entry.
     */
    public boolean isValid();

    /**
     * Frees resources associated with this entry. The entry is not valid after this call anymore.
     * Depending on the value of <code>clearPersist</code> and the persistence status of the entry,
     * the persistent part of the entry is freed/deleted as well
     *
     * @param clearPersist Set to true, if this entry should be deleted finally, even if the persist flag is set.
     */
    public void clear(boolean clearPersist);

    /**
     * @return The hash value of this entry
     */
    public String getHash();

    /**
     * @return The result properties of this entry;
     */
    public HashMap<String, Object> getResultProperties();

    /**
     * @return The local filesystem directory, where the entries persistent data is stored
     */
    public File getPersistentEntryDirectory();

    /**
     * @return The size of the entries persistent directory on its local file system
     */
    public long getPersistentSize();

    /**
     * Locks the entry, so that it won't be removed from the cache.
     */
    public void acquireLock();

    /**
     * Releases the entry lock, so that it can be removed from the cache.
     */
    public void releaseLock();

    /**
     * @return True if this object is currently locked
     */
    public boolean isLocked();

    /**
     * Updates the entries last access time.
     */
    public void touch();

    /**
     * @return The point of time, when the entry was accessed the last time
     */
    public Date getAccessTime();

    /**
     * Makes the data of this entry persistent, so that it can be completely recreated at a later point of time; if the entry has been made
     * persistent before and doesn't need an update, the method returns true, also.
     *
     * @return true, if the file has been made persistent (even in a previous call)
     */
    boolean makePersistent(ICache associatedCache);

    /**
     * @return true, if the entry has been made persistent before, not changing its content anymore
     */
    boolean isPersistent();
}
