/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.api;

import static com.openexchange.obs.api.Architecture.i586;
import static com.openexchange.obs.api.Architecture.x86_64;
import java.util.ArrayList;
import java.util.List;

/**
 * {@link Repository}
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class Repository {

    public static final Repository STANDARD_I586 = new Repository("standard", i586);
    public static final Repository STANDARD_X86_64 = new Repository("standard", x86_64);

    private final String name;
    private List<Path> path = new ArrayList<Path>();
    private final Architecture arch;

    public Repository(String name, Architecture arch) {
        this(name, null, arch);
    }

    public Repository(String name, Path path, Architecture arch) {
        super();
        this.name = name;
        if (path != null) {
            this.path.add(path);
        }
        this.arch = arch;
    }

    public String getName() {
        return name;
    }

    public Path[] getPaths() {
        return path.toArray(new Path[path.size()]);
    }

    public Architecture getArch() {
        return arch;
    }

    public void addPath(Path myPath) {
        this.path.add(myPath);
    }

    public void addPaths(final Path[] paths){
        for (Path p : paths){
            this.path.add(p);
        }
    }

    public static Repository getNewRepository(Repositories repo, Architecture arch) throws BuildServiceException {
        switch (repo) {
        case standard:
            return standard(arch);
        case CentOS5:
            return new Repository("CentOS5", new Path(standard(arch), Project.CENTOS_5), arch);
        case DebianLenny:
            return new Repository("DebianLenny", new Path(standard(arch), Project.DEBIAN_LENNY), arch);
        case DebianSqueeze:
            return new Repository("DebianSqueeze", new Path(standard(arch), Project.DEBIAN_SQUEEZE), arch);
        case RHEL5:
            return new Repository("RHEL5", new Path(standard(arch), Project.RHEL_5), arch);
        case RHEL6:
            return new Repository("RHEL6", new Path(standard(arch), Project.RHEL_6), arch);
        case SLES11:
            return new Repository("SLES11", new Path(standard(arch), Project.SLES_11), arch);
        default:
            throw new BuildServiceException("Unknown repository: \"" + repo + "\".");
        }
    }

    private static Repository standard(Architecture arch) throws BuildServiceException {
        switch (arch) {
        case i586:
            return Repository.STANDARD_I586;
        case x86_64:
            return Repository.STANDARD_X86_64;
        default:
            throw new BuildServiceException("Unknown architecture: \"" + arch + "\".");
        }
    }

    public static Repository getNewRepository(Repositories repo, Architecture arch, String[] dependingProjectNames) throws BuildServiceException {
        if (dependingProjectNames != null) {
            Repository tmpRepository = new Repository(repo.name(), arch);
            for (String s : dependingProjectNames){
                tmpRepository.addPath(new Path(Repository.getNewRepository(repo, arch), new Project(s)));
            }
            return tmpRepository;
        }
        return getNewRepository(repo, arch);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(name + " " + arch.toString() + " ");
        for (Path p : getPaths()){
            sb.append(p.toString());
        }
        return sb.toString();
    }
}
